<?php
/**
 * Theme functions and definitions
 *
 * @package HelloElementor
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

define( 'HELLO_ELEMENTOR_VERSION', '3.4.4' );
define( 'EHP_THEME_SLUG', 'hello-elementor' );

define( 'HELLO_THEME_PATH', get_template_directory() );
define( 'HELLO_THEME_URL', get_template_directory_uri() );
define( 'HELLO_THEME_ASSETS_PATH', HELLO_THEME_PATH . '/assets/' );
define( 'HELLO_THEME_ASSETS_URL', HELLO_THEME_URL . '/assets/' );
define( 'HELLO_THEME_SCRIPTS_PATH', HELLO_THEME_ASSETS_PATH . 'js/' );
define( 'HELLO_THEME_SCRIPTS_URL', HELLO_THEME_ASSETS_URL . 'js/' );
define( 'HELLO_THEME_STYLE_PATH', HELLO_THEME_ASSETS_PATH . 'css/' );
define( 'HELLO_THEME_STYLE_URL', HELLO_THEME_ASSETS_URL . 'css/' );
define( 'HELLO_THEME_IMAGES_PATH', HELLO_THEME_ASSETS_PATH . 'images/' );
define( 'HELLO_THEME_IMAGES_URL', HELLO_THEME_ASSETS_URL . 'images/' );

if ( ! isset( $content_width ) ) {
	$content_width = 800; // Pixels.
}

if ( ! function_exists( 'hello_elementor_setup' ) ) {
	/**
	 * Set up theme support.
	 *
	 * @return void
	 */
	function hello_elementor_setup() {
		if ( is_admin() ) {
			hello_maybe_update_theme_version_in_db();
		}

		if ( apply_filters( 'hello_elementor_register_menus', true ) ) {
			register_nav_menus( [ 'menu-1' => esc_html__( 'Header', 'hello-elementor' ) ] );
			register_nav_menus( [ 'menu-2' => esc_html__( 'Footer', 'hello-elementor' ) ] );
		}

		if ( apply_filters( 'hello_elementor_post_type_support', true ) ) {
			add_post_type_support( 'page', 'excerpt' );
		}

		if ( apply_filters( 'hello_elementor_add_theme_support', true ) ) {
			add_theme_support( 'post-thumbnails' );
			add_theme_support( 'automatic-feed-links' );
			add_theme_support( 'title-tag' );
			add_theme_support(
				'html5',
				[
					'search-form',
					'comment-form',
					'comment-list',
					'gallery',
					'caption',
					'script',
					'style',
					'navigation-widgets',
				]
			);
			add_theme_support(
				'custom-logo',
				[
					'height'      => 100,
					'width'       => 350,
					'flex-height' => true,
					'flex-width'  => true,
				]
			);
			add_theme_support( 'align-wide' );
			add_theme_support( 'responsive-embeds' );

			/*
			 * Editor Styles
			 */
			add_theme_support( 'editor-styles' );
			add_editor_style( 'editor-styles.css' );

			/*
			 * WooCommerce.
			 */
			if ( apply_filters( 'hello_elementor_add_woocommerce_support', true ) ) {
				// WooCommerce in general.
				add_theme_support( 'woocommerce' );
				// Enabling WooCommerce product gallery features (are off by default since WC 3.0.0).
				// zoom.
				add_theme_support( 'wc-product-gallery-zoom' );
				// lightbox.
				add_theme_support( 'wc-product-gallery-lightbox' );
				// swipe.
				add_theme_support( 'wc-product-gallery-slider' );
			}
		}
	}
}
add_action( 'after_setup_theme', 'hello_elementor_setup' );

function hello_maybe_update_theme_version_in_db() {
	$theme_version_option_name = 'hello_theme_version';
	// The theme version saved in the database.
	$hello_theme_db_version = get_option( $theme_version_option_name );

	// If the 'hello_theme_version' option does not exist in the DB, or the version needs to be updated, do the update.
	if ( ! $hello_theme_db_version || version_compare( $hello_theme_db_version, HELLO_ELEMENTOR_VERSION, '<' ) ) {
		update_option( $theme_version_option_name, HELLO_ELEMENTOR_VERSION );
	}
}

if ( ! function_exists( 'hello_elementor_display_header_footer' ) ) {
	/**
	 * Check whether to display header footer.
	 *
	 * @return bool
	 */
	function hello_elementor_display_header_footer() {
		$hello_elementor_header_footer = true;

		return apply_filters( 'hello_elementor_header_footer', $hello_elementor_header_footer );
	}
}

if ( ! function_exists( 'hello_elementor_scripts_styles' ) ) {
	/**
	 * Theme Scripts & Styles.
	 *
	 * @return void
	 */
	function hello_elementor_scripts_styles() {
		if ( apply_filters( 'hello_elementor_enqueue_style', true ) ) {
			wp_enqueue_style(
				'hello-elementor',
				HELLO_THEME_STYLE_URL . 'reset.css',
				[],
				HELLO_ELEMENTOR_VERSION
			);
		}

		if ( apply_filters( 'hello_elementor_enqueue_theme_style', true ) ) {
			wp_enqueue_style(
				'hello-elementor-theme-style',
				HELLO_THEME_STYLE_URL . 'theme.css',
				[],
				HELLO_ELEMENTOR_VERSION
			);
		}

		if ( hello_elementor_display_header_footer() ) {
			wp_enqueue_style(
				'hello-elementor-header-footer',
				HELLO_THEME_STYLE_URL . 'header-footer.css',
				[],
				HELLO_ELEMENTOR_VERSION
			);
		}
	}
}
add_action( 'wp_enqueue_scripts', 'hello_elementor_scripts_styles' );

if ( ! function_exists( 'hello_elementor_register_elementor_locations' ) ) {
	/**
	 * Register Elementor Locations.
	 *
	 * @param ElementorPro\Modules\ThemeBuilder\Classes\Locations_Manager $elementor_theme_manager theme manager.
	 *
	 * @return void
	 */
	function hello_elementor_register_elementor_locations( $elementor_theme_manager ) {
		if ( apply_filters( 'hello_elementor_register_elementor_locations', true ) ) {
			$elementor_theme_manager->register_all_core_location();
		}
	}
}
add_action( 'elementor/theme/register_locations', 'hello_elementor_register_elementor_locations' );

if ( ! function_exists( 'hello_elementor_content_width' ) ) {
	/**
	 * Set default content width.
	 *
	 * @return void
	 */
	function hello_elementor_content_width() {
		$GLOBALS['content_width'] = apply_filters( 'hello_elementor_content_width', 800 );
	}
}
add_action( 'after_setup_theme', 'hello_elementor_content_width', 0 );

if ( ! function_exists( 'hello_elementor_add_description_meta_tag' ) ) {
	/**
	 * Add description meta tag with excerpt text.
	 *
	 * @return void
	 */
	function hello_elementor_add_description_meta_tag() {
		if ( ! apply_filters( 'hello_elementor_description_meta_tag', true ) ) {
			return;
		}

		if ( ! is_singular() ) {
			return;
		}

		$post = get_queried_object();
		if ( empty( $post->post_excerpt ) ) {
			return;
		}

		echo '<meta name="description" content="' . esc_attr( wp_strip_all_tags( $post->post_excerpt ) ) . '">' . "\n";
	}
}
add_action( 'wp_head', 'hello_elementor_add_description_meta_tag' );

// Settings page
require get_template_directory() . '/includes/settings-functions.php';

// Header & footer styling option, inside Elementor
require get_template_directory() . '/includes/elementor-functions.php';

if ( ! function_exists( 'hello_elementor_customizer' ) ) {
	// Customizer controls
	function hello_elementor_customizer() {
		if ( ! is_customize_preview() ) {
			return;
		}

		if ( ! hello_elementor_display_header_footer() ) {
			return;
		}

		require get_template_directory() . '/includes/customizer-functions.php';
	}
}
add_action( 'init', 'hello_elementor_customizer' );

if ( ! function_exists( 'hello_elementor_check_hide_title' ) ) {
	/**
	 * Check whether to display the page title.
	 *
	 * @param bool $val default value.
	 *
	 * @return bool
	 */
	function hello_elementor_check_hide_title( $val ) {
		if ( defined( 'ELEMENTOR_VERSION' ) ) {
			$current_doc = Elementor\Plugin::instance()->documents->get( get_the_ID() );
			if ( $current_doc && 'yes' === $current_doc->get_settings( 'hide_title' ) ) {
				$val = false;
			}
		}
		return $val;
	}
}
add_filter( 'hello_elementor_page_title', 'hello_elementor_check_hide_title' );

/**
 * BC:
 * In v2.7.0 the theme removed the `hello_elementor_body_open()` from `header.php` replacing it with `wp_body_open()`.
 * The following code prevents fatal errors in child themes that still use this function.
 */
if ( ! function_exists( 'hello_elementor_body_open' ) ) {
	function hello_elementor_body_open() {
		wp_body_open();
	}
}

require HELLO_THEME_PATH . '/theme.php';

HelloTheme\Theme::instance();




/*
 * کد دوره
 * */

add_action('add_meta_boxes', function () {
    global $post;

    // فقط برای محصولات
    if ($post->post_type !== 'product') return;

    // گرفتن تمام دسته‌های این محصول
    $terms = get_the_terms($post->ID, 'product_cat');
    if (!$terms || is_wp_error($terms)) return;

    // آیدی تمام دسته‌ها (برای بررسی سلسله‌مراتب)
    $term_ids = wp_list_pluck($terms, 'term_id');

    // بررسی اینکه آیا یکی از دسته‌ها یا پدرانشان برابر ۳۳ است
    $apply_box = false;

    foreach ($term_ids as $term_id) {
        if (has_term(31, 'product_cat', $post)) {
            $apply_box = true;
            break;
        }

        // بررسی پدر دسته
        $parents = get_ancestors($term_id, 'product_cat');
        if (in_array(31, $parents)) {
            $apply_box = true;
            break;
        }
    }

    if ($apply_box) {
        add_meta_box(
            'course_contents_metabox',
            'محتویات دوره',
            'render_course_metabox',
            'product',
            'normal',
            'high'
        );
    }
});




function render_course_metabox($post) {
    $contents = get_post_meta($post->ID, '_course_contents', true);
    ?>
    <div id="course-contents-wrapper">
        <button type="button" class="button add-topic">+ افزودن موضوع</button>
        <div id="course-contents">
            <?php if (!empty($contents)) foreach ($contents as $ti => $topic) { ?>
                <div class="course-topic" data-index="<?php echo $ti; ?>">
                    <input type="text" name="course_contents[<?php echo $ti; ?>][title]" value="<?php echo esc_attr($topic['title']); ?>" placeholder="عنوان موضوع">
                    <button type="button" class="button add-lesson">+ افزودن جلسه</button>
                    <button type="button" class="button remove-topic">حذف موضوع</button>
                    <div class="lessons">
                        <?php if (!empty($topic['lessons'])) foreach ($topic['lessons'] as $li => $lesson) { ?>
                            <div class="lesson-item">
                                <input type="text" name="course_contents[<?php echo $ti; ?>][lessons][<?php echo $li; ?>][title]" value="<?php echo esc_attr($lesson['title']); ?>" placeholder="عنوان جلسه">
                                <input type="text" name="course_contents[<?php echo $ti; ?>][lessons][<?php echo $li; ?>][attachments]" value="<?php echo esc_attr($lesson['attachments']); ?>" placeholder="متعلقات">
                                <?php wp_editor($lesson['content'], 'editor_' . $ti . '_' . $li, [
                                    'textarea_name' => "course_contents[{$ti}][lessons][{$li}][content]",
                                    'textarea_rows' => 5,
                                ]); ?>
                                <button type="button" class="button remove-lesson">حذف جلسه</button>
                            </div>
                        <?php } ?>
                    </div>
                </div>
            <?php } ?>
        </div>
    </div>
    <style>
        .course-topic, .lesson-item { border: 1px solid #ccc; margin: 10px 0; padding: 10px; background: #f9f9f9; }
        .lesson-item { margin-right: 20px; }
        .remove-topic, .remove-lesson { background: #d63638; color: white; margin-top: 5px; }
    </style>
    <script>
        jQuery(function($) {
            let topicIndex = $('#course-contents .course-topic').length;

            $('.add-topic').click(function () {
                const newTopic = `<div class="course-topic" data-index="${topicIndex}">
                    <input type="text" name="course_contents[${topicIndex}][title]" placeholder="عنوان موضوع">
                    <button type="button" class="button add-lesson">+ افزودن جلسه</button>
                    <button type="button" class="button remove-topic">حذف موضوع</button>
                    <div class="lessons"></div>
                </div>`;
                $('#course-contents').append(newTopic);
                topicIndex++;
            });

            $('#course-contents').on('click', '.remove-topic', function () {
                $(this).closest('.course-topic').remove();
            });

            $('#course-contents').on('click', '.add-lesson', function () {
                const topic = $(this).closest('.course-topic');
                const ti = topic.data('index');
                const lessonCount = topic.find('.lesson-item').length;
                const newLesson = `<div class="lesson-item">
                    <input type="text" name="course_contents[${ti}][lessons][${lessonCount}][title]" placeholder="عنوان جلسه">
                    <input type="text" name="course_contents[${ti}][lessons][${lessonCount}][attachments]" placeholder="متعلقات">
                    <textarea name="course_contents[${ti}][lessons][${lessonCount}][content]"></textarea>
                    <button type="button" class="button remove-lesson">حذف جلسه</button>
                </div>`;
                topic.find('.lessons').append(newLesson);
            });

            $('#course-contents').on('click', '.remove-lesson', function () {
                $(this).closest('.lesson-item').remove();
            });
        });
    </script>
    <?php
}

// 2. ذخیره متا
add_action('save_post', function($post_id) {
    if (isset($_POST['course_contents'])) {
        update_post_meta($post_id, '_course_contents', $_POST['course_contents']);
    }
});




// شورت‌کد برای نمایش محتویات دوره در صفحه محصول
add_shortcode('course_contents', function () {
    global $post;
    if ($post->post_type !== 'product') return '';

    $contents = get_post_meta($post->ID, '_course_contents', true);
    if (!$contents || !is_array($contents)) return '<p>محتوایی برای این دوره ثبت نشده است.</p>';

    ob_start();
    ?>
    <div class="course-content-wrapper">
        <h2 class="course-title">محتویات دوره</h2>
        <?php foreach ($contents as $topic_index => $topic): ?>
            <div class="topic-block">
                <h3 class="topic-title"><?php echo esc_html($topic['title']); ?></h3>
                <div class="lessons-wrapper">
                    <?php foreach ($topic['lessons'] as $lesson_index => $lesson): ?>
                        <div class="lesson-item">
                            <button class="accordion-toggle">📹 <?php echo esc_html($lesson['title']); ?></button>
                            <div class="lesson-content">
                                <?php if (!empty($lesson['attachments'])): ?>
                                    <p><strong>متعلقات:</strong> <?php echo esc_html($lesson['attachments']); ?></p>
                                <?php endif; ?>
                                <div class="lesson-text">
                                    <?php echo wpautop($lesson['content']); ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endforeach; ?>
    </div>

    <style>
   .course-content-wrapper {
    font-family: 'IRANYekanFN';
   
}
    .course-title { font-size: 24px; font-weight: bold; margin-bottom: 20px; border-bottom: 2px solid #eee; padding-bottom: 10px; }
    .topic-block { margin-bottom: 25px; }
    .topic-title {font-size: 20px;
    color: #a12b28;
    margin-bottom: 10px; }
    .lesson-item { margin-bottom: 10px; }
    .accordion-toggle {
       background-color: #f7e9d2;
       color: #394352;;
        cursor: pointer;
        padding: 12px;
        width: 100%;
        border: none;
        text-align: right;
        font-size: 16px;
        border-radius: 5px;
        transition: background 0.3s;
    }
		
		.accordion-toggle:hover , .accordion-toggle:focus {
			  background-color: #ffd5a4;
       color: #394352;;
		}
    .accordion-toggle:hover { background-color: #e0e0e0; }
    .lesson-content {
        display: none;
        padding: 15px;
        background-color: #fafafa;
        border: 1px solid #ddd;
        border-radius: 5px;
        margin-top: 5px;
    }
    .lesson-text { margin-top: 10px; line-height: 1.7; }
    </style>

    <script>
    document.addEventListener("DOMContentLoaded", function () {
        const toggles = document.querySelectorAll(".accordion-toggle");
        toggles.forEach(toggle => {
            toggle.addEventListener("click", function () {
                const content = this.nextElementSibling;
                content.style.display = content.style.display === "block" ? "none" : "block";
            });
        });
    });
    </script>
    <?php
    return ob_get_clean();
});



